#include <chrono>
#include <iostream>
#include <string>
#include <stdio.h>
#include <random>
#include <cstring>
#include "external/MurmurHash3.h"
#include <bitset>
#include <immintrin.h>
#include <bits/stdc++.h>
#include <thread>

template <typename T>
class ThreeCoord {
public:
  T x;
  T y;
  T z;

  ThreeCoord() {}

  ThreeCoord(T _x, T _y, T _z) {
    x = _x;
    y = _y;
    z = _z;
  }

  operator bool() const
  {
    return (x > 0 && y > 0 && z > 0); 
  }

  int vol() {
    return x * y * z;
  }
};

ThreeCoord<int> corridorToFull(ThreeCoord<int> corridorCursor){
  return ThreeCoord<int>(2 * corridorCursor.x + 1, 2 * corridorCursor.y + 1, 2 * corridorCursor.z + 1);
}

ThreeCoord<int> corridorToFull(int x, int y, int z)
{
  return ThreeCoord<int>(2 * x + 1, 2 * y + 1, 2 * z + 1);
}

int access(ThreeCoord<int> cursor, ThreeCoord<int> d){
  return ((cursor.z) * d.y * d.x) + ((cursor.y) * d.x) + (cursor.x);
}

int access(int x, int y, int z, ThreeCoord<int> d)
{
  return ((z) * d.y * d.x) + ((y) * d.x) + (x);
}

uint8_t availableLocations(uint8_t* maze, ThreeCoord<int> d, ThreeCoord<int> cursor)
{
  /*
  na na Z- Z+ Y- Y+ X- X+
  origin *
                        Y-
           . *------------------------+
         .'  |                      .'|
       .'    |                   .-'  |
      +------+-----------------+'     |
      |      |                 |      |
      |      |                 |      |
      |      |           Z-    |      | X+
      |      |                 |      |
    X-|      |   Z+            |      |
      |      |                 |      |
      |      |                 |      |
      |      +-----------------+------+
      |    .'                  |    .'
      |  .'                    |  .'
      +------------------------+'
                  Y+
                    -Y -Z
                     |/
                -X---+---X+
                    /|
                  +Z +Y
*/

  uint8_t flags{0b00000000};
  // DIR_RIGHT = 1 << 0 X+
  // DIR_LEFT = 1 << 1  X-
  // DIR_UP = 1 << 2    Y+
  // DIR_DOWN = 1 << 3  Y-
  // DIR_FWD = 1 << 4   Z+
  // DIR_BACK = 1 << 5  Z-
  // flags |= (isAvailable << i);
  if (cursor.x < d.x - 2)
  {
    int index = access(cursor.x + 2, cursor.y, cursor.z, d);
    flags |= (!(*(maze+index)) << 0); // X+
  }
  if (cursor.x >= 2)
  {
    int index = access(cursor.x - 2, cursor.y, cursor.z, d);
    flags |= (!(*(maze + index)) << 1); // X-
  }

  if (cursor.y < d.y - 2)
  {
    int index = access(cursor.x, cursor.y + 2, cursor.z, d);
    flags |= (!(*(maze + index)) << 2); // Y+
  }
  if (cursor.y >= 2)
  {
    int index = access(cursor.x, cursor.y - 2, cursor.z, d);
    flags |= (!(*(maze + index)) << 3); // Y-
  }

  if (cursor.z < d.z - 2)
  {
    int index = access(cursor.x, cursor.y, cursor.z + 2, d);
    flags |= (!(*(maze + index)) << 4); // Z+
  }
  if (cursor.z >= 2)
  {
    int index = access(cursor.x, cursor.y, cursor.z - 2, d);
    flags |= (!(*(maze + index)) << 5); // Z-
  }
  return flags;
}

void printFlatMaze(uint8_t *maze, ThreeCoord<int> d, ThreeCoord<int> cursor)
{
  for (int y = 0; y < d.y; y++)
  {
    for (int z = 0; z < d.z; z++)
    {
      for (int x = 0; x < d.x; x++)
      {
        int index = x + y * d.x + z * d.x * d.y;
        if (cursor.x == x && cursor.y == y && cursor.z == z)
        {
          printf("X ");
        }
        else
        {
          printf("%d ", *(maze + index));
        }
      }
      printf("  "); // space between z-slices
    }
    printf("\n"); // line per Y-row
  }
}

int getDirection(int dir, int offset){
  return (2 * (dir - offset)) - 1;
}

int translateDirection(uint8_t available, int chosenDirection)
{
  int i = 0;
  int j = 0;
  while (i < 6) {
    if ((available & (1 << i)) > 0) {
      j++;
      if (j >= chosenDirection) {
        return i;
      }
    }
    i++;
  }
  return i;
}

ThreeCoord<int> step(uint8_t *maze, ThreeCoord<int> cursor, ThreeCoord<int> d, std::mt19937 gen, std::uniform_int_distribution<> dis)
{
  //printf("Cursor before: (%d, %d, %d)\n", cursor.x, cursor.y, cursor.z);
  uint8_t available = availableLocations(maze, d, cursor); // test available locations at cursor
  uint8_t popCount = _mm_popcnt_u32((uint32_t)available); // total options available
  if (!popCount) {
    //printf("No Available Moves\n");
    return ThreeCoord<int>(0, 0, 0); // no available moves
  }
  //printf("Total options: %d\n", popCount);
  uint8_t chosenDirection = dis(gen) % popCount; // random number 0 - popCount-1
  
  int i = translateDirection(available, chosenDirection);
  int delta = 0;
  //printf("i: %d\nChosen Direction: %d\n", i, chosenDirection);
  if (i < 2)
  {
    *(maze + access(cursor.x - getDirection(i, 0), cursor.y, cursor.z, d)) = 1; // open the wall
    delta = getDirection(i, 0) * -2;
    //printf("moving cursor along x axis by %d\n", delta);
    cursor.x += delta; // move the cursor
  }
  else if (i < 4)
  {
    *(maze + access(cursor.x, cursor.y - getDirection(i, 2), cursor.z, d)) = 1; // open the wall
    delta = getDirection(i, 2) * -2;
    //printf("moving cursor along y axis by %d\n", delta);
    cursor.y += delta; // move the cursor
  }
  else if (i < 6)
  {
    *(maze + access(cursor.x, cursor.y, cursor.z - getDirection(i, 4), d)) = 1; // open the wall
    delta = getDirection(i, 4) * -2;
    //printf("moving cursor along z axis by %d\n", delta);
    cursor.z += delta; // move the cursor
  } else {
    printf("\n\n\nFAIL\n\n\n");
  }

  *(maze + access(cursor, d)) = 1;

  return cursor;
}

    // ThreeCoord<int> choose(ThreeCoord<int> d, ThreeCoord<int> cursor, uint8_t available){
    //   ThreeCoord<int> next;
    //   uint8_t popCount = _mm_popcnt_u32((uint32_t)available);
    // }

ThreeCoord<int> halls;



int main(int argc, char *argv[])
{
  // Ensure
  if (!(argc == 4 || argc == 5)) {
    printf("argc: %d\n", argc);
    printf("Usage: %s <int x_halls> <int y_halls> <int z_halls> <seed>\n", argv[0]);
    return 1;
  }
  
  try
  {
    halls.x = std::stoi(argv[1]);
    halls.y = std::stoi(argv[2]);
    halls.z = std::stoi(argv[3]);
  }
  catch (const std::invalid_argument &e)
  {
    printf("Invalid argument: %s\n", e.what());
    printf("Usage: %s <int x_halls> <int y_halls> <int z_halls> <seed>\n", argv[0]);
    return 1;
  }
  catch (const std::out_of_range &e)
  {
    printf("Out of range: %s\n", e.what());
    printf("Usage: %s <int x_halls> <int y_halls> <int z_halls> <seed>\n", argv[0]);
    return 1;
  }
  
  
  std::random_device rd;  // Obtain a random number from hardware
  std::mt19937 gen;       // merser twister engine

  if (argc != 5)
  {
    printf("Using random seed.\n");
    gen = std::mt19937(rd());
  }
  else
  {
    printf("Hashing input to create seed.\n");
    uint32_t seed;
    MurmurHash3_x86_32(argv[4],std::strlen(argv[4]),0,&seed); // Use murmurHash3_x86_32 for consistency across platforms, see readme
    gen = std::mt19937(seed); // Nice clean deterministic generator
    printf("Seed: %u\n", seed);
  }

  std::uniform_int_distribution<> dis(0, 59); // 60 is lowest common multiple of 1-6, 0-59 for 60 options, only one generator => avoid branching

  // W H W H W
  // D.x = 5, D.y = 5
  // 0 0 0 0 0   W
  // 0 2 1 2 0   H
  // 0 1 0 1 0   W
  // 0 2 0 2 0   H
  // 0 0 0 0 0   W

  // Every corridor (where a vertical and horizontal hall cross, marked 2) will always be open at the end
  // Every wall cornered by a 2 will always be closed
  // Walls between ( Vertical/Horizontal ) can be open or closed
  // A maze can be considered perfect if every pair of points is reachable through one path
  // A perfect maze can be generated by starting a random walk from any previously visited cell
  // Walls are opened one step at a time

  // Width = halls * 2 + 1
  
  ThreeCoord<int> d = ThreeCoord((halls.x * 2) + 1, (halls.y * 2) + 1, (halls.z * 2) + 1);
  
  // Create a uint8_t maze array
  uint8_t *maze = new uint8_t[d.vol()];
  // Ensure that maze is initialized to zero
  std::memset(maze, 0, d.vol());

  // Start in the corner
  ThreeCoord<int> cursor = ThreeCoord(halls.x / 2, halls.y / 2, halls.z / 2);
  cursor.x = (cursor.x * 2) + 1;
  cursor.y = (cursor.y * 2) + 1;
  cursor.z = (cursor.z * 2) + 1;
  *(maze + access(cursor, d)) = 1; // init maze

  auto t1 = std::chrono::high_resolution_clock::now();
  // Each corridor requires one transition in, and with one initialized corridor, there will be exactly (x * y * z) - 1 steps
  for (int i = 0; i < halls.vol() - 1; i++){
    // printf("%d/%d\n",i,halls.vol());
    if (cursor && (i % 10) != 9){
      cursor = step(maze, cursor, d, gen, dis);
    } else {
      bool done = false;
      for (int x = 0, y = 0, z = 0; z < halls.z && !done;){
        if (*(maze + access(corridorToFull(x, y, z), d)) == 1 && availableLocations(maze, d, corridorToFull(ThreeCoord<int>(x, y, z)))){ // CONVERT TO FULL, NOT JUST CORRIDOR
          // IF hall at (x, y, z) 
          done = true;
          cursor = corridorToFull(x, y, z); // HERE TOO
          cursor = step(maze, cursor, d, gen, dis);
        }
        x++;
        if (x >= halls.x)
        {
          x = 0;
          y++;
          if (y >= halls.y)
          {
            y = 0;
            z++;
          }
        }
      }
    }
  }
  auto t2 = std::chrono::high_resolution_clock::now();
  std::chrono::duration<float, std::milli> fp_ms = t2 - t1;
  printf("Time is: %f ms\n", fp_ms.count());
  // printFlatMaze(maze, d, cursor);

  std::ofstream fileout("maze.mze");
  fileout << d.x << d.y << d.z;
  for (int i = 0; i < d.vol(); i++){
    fileout << *(maze + i);
  }

  delete[] maze;
  return 0;
}

// gcc mazegen.cc -o mazegen.x

// ./mazegen.x x y z seed

// TODO: Rewrite maze format to pack 4-bit VRBD flags per corridor using 32-bit ints for 8-way compressed spatial encoding


